<?php
/**
 * Class Categories_API file.
 *
 * @package WebToffee
 */

namespace WebToffee\CookieConsent\Lite\Admin\Modules\Cookies\Api;

use WP_REST_Server;
use WebToffee\CookieConsent\Lite\Admin\Modules\Cookies\Api\API_Controller;
use WebToffee\CookieConsent\Lite\Admin\Modules\Cookies\Includes\Cookie_Categories;
use WebToffee\CookieConsent\Lite\Admin\Modules\Cookies\Includes\Category_Controller;
use WP_Error;
use Exception;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Cookie categories API
 *
 * @class       Categories_API
 * @version     3.0.0
 * @package     WebToffee
 * @extends     API_Controller
 */
class Categories_API extends API_Controller {

	/**
	 * Endpoint namespace.
	 *
	 * @var string
	 */
	protected $namespace = 'wcc/v1';

	/**
	 * Route base.
	 *
	 * @var string
	 */
	protected $rest_base = 'cookies/categories';

	/**
	 * Constructor
	 */
	public function __construct() {
		add_action( 'rest_api_init', array( $this, 'register_routes' ), 10 );
	}
	/**
	 * Register the routes for cookies.
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base,
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_items' ),
					'permission_callback' => array( $this, 'get_items_permissions_check' ),
					'args'                => $this->get_collection_params(),
				),
				array(
					'methods'             => WP_REST_Server::CREATABLE,
					'callback'            => array( $this, 'create_item' ),
					'permission_callback' => array( $this, 'create_item_permissions_check' ),
					'args'                => $this->get_endpoint_args_for_item_schema( WP_REST_Server::CREATABLE ),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/bulk',
			array(
				array(
					'methods'             => WP_REST_Server::CREATABLE,
					'callback'            => array( $this, 'bulk' ),
					'permission_callback' => array( $this, 'create_item_permissions_check' ),
					'args'                => $this->get_endpoint_args_for_item_schema( WP_REST_Server::CREATABLE ),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)',
			array(
				'args'   => array(
					'id' => array(
						'description' => __( 'Unique identifier for the resource.', 'webtoffee-cookie-consent' ),
						'type'        => 'integer',
					),
				),
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_item' ),
					'permission_callback' => array( $this, 'get_item_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'update_item' ),
					'permission_callback' => array( $this, 'update_item_permissions_check' ),
					'args'                => $this->get_endpoint_args_for_item_schema( WP_REST_Server::EDITABLE ),
				),
				array(
					'methods'             => WP_REST_Server::DELETABLE,
					'callback'            => array( $this, 'delete_item' ),
					'permission_callback' => array( $this, 'delete_item_permissions_check' ),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/nocookiescripts',
			array(
				array(
					'methods'             => WP_REST_Server::CREATABLE,
					'callback'            => array( $this, 'update_no_cookie_script' ),
					'permission_callback' => array( $this, 'create_item_permissions_check' ),
					'args'                => $this->get_endpoint_args_for_item_schema( WP_REST_Server::CREATABLE ),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);
	}
	/**
	 * Return cookie ids
	 *
	 * @param array $args Request arguments.
	 * @return array
	 */
	public function get_item_objects( $args ) {
		return Category_Controller::get_instance()->get_items( $args );
	}

	/**
	 * Return item object
	 *
	 * @param object $item Cookie item.
	 * @return Cookie_Categories
	 */
	public function get_item_object( $item = false ) {
		return new Cookie_Categories( $item );
	}
	/**
	 * Get formatted item data.
	 *
	 * @since  3.0.0
	 * @param  Cookie_Categories $cookie_category Cookie instance.
	 * @return array
	 */
	protected function get_formatted_item_data( $cookie_category ) {
		return array(
			'id'                 => $cookie_category->get_id(),
			'name'               => $cookie_category->get_name(),
			'slug'               => $cookie_category->get_slug(),
			'description'        => $cookie_category->get_description(),
			'prior_consent'      => $cookie_category->get_prior_consent(),
			'visibility'         => $cookie_category->get_visibility(),
			'language'           => $cookie_category->get_language(),
			'priority'           => $cookie_category->get_priority(),
			'sell_personal_data' => $cookie_category->get_sell_personal_data(),
			'cookie_list'        => $cookie_category->get_cookies(),
			'date_created'       => $cookie_category->get_date_created(),
			'date_modified'      => $cookie_category->get_date_modified(),
		);
	}
	/**
	 * Bulk update of categories.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 * @return WP_Error|WP_REST_Response
	 */
	public function bulk( $request ) {

		try {
			if ( ! isset( $request['categories'] ) ) {
				return new WP_Error( 'webtoffee_rest_invalid_data', __( 'No data specified to create/edit categories', 'webtoffee-cookie-consent' ), array( 'status' => 404 ) );
			}
			if ( ! defined( 'WCC_BULK_REQUEST' ) ) {
				define( 'WCC_BULK_REQUEST', true );
			}
			$item_objects = array();
			$objects      = array();
			$data         = $request['categories'];

			$categories = array();

			foreach ( $data as $_category ) {
				$id = 0;
				// Try to get the category ID.
				if ( isset( $_category['id'] ) ) {
					$id = intval( $_category['id'] );
				}

				if ( $id ) {
					$object        = new Cookie_Categories( $id );
					$prior_consent = isset( $_category['prior_consent'] ) ? (bool) $_category['prior_consent'] : false;
					$visibility    = isset( $_category['visibility'] ) ? (bool) $_category['visibility'] : true;
					$object->set_prior_consent( $prior_consent );
					$object->set_visibility( $visibility );
					$object->save();
					$item_objects[] = $object;

				}
			}
			foreach ( $item_objects as $data ) {
				$data      = $this->prepare_item_for_response( $data, $request );
				$objects[] = $this->prepare_response_for_collection( $data );
			}
			do_action( 'wcc_after_update_cookie_category' );
			return rest_ensure_response( $objects );
		} catch ( Exception $e ) {
			return new WP_Error( $e->getCode(), $e->getMessage(), array( 'status' => $e->getCode() ) );
		}
	}
	/**
	 * Get the Cookies's schema, conforming to JSON Schema.
	 *
	 * @return array
	 */
	public function get_item_schema() {
		$schema = array(
			'$schema'    => 'http://json-schema.org/draft-04/schema#',
			'title'      => 'cookie_categories',
			'type'       => 'object',
			'properties' => array(
				'id'                 => array(
					'description' => __( 'Unique identifier for the resource.', 'webtoffee-cookie-consent' ),
					'type'        => 'integer',
					'context'     => array( 'view' ),
					'readonly'    => true,
				),
				'date_created'       => array(
					'description' => __( 'The date the cookie was created, as GMT.', 'webtoffee-cookie-consent' ),
					'type'        => 'date-time',
					'context'     => array( 'view', 'edit' ),
					'readonly'    => true,
				),
				'date_modified'      => array(
					'description' => __( 'The date the cookie was last modified, as GMT.', 'webtoffee-cookie-consent' ),
					'type'        => 'date-time',
					'context'     => array( 'view', 'edit' ),
					'readonly'    => true,
				),
				'name'               => array(
					'description' => __( 'Cookie category name.', 'webtoffee-cookie-consent' ),
					'type'        => 'object',
					'context'     => array( 'view', 'edit' ),
				),
				'slug'               => array(
					'description' => __( 'Cookie category unique name', 'webtoffee-cookie-consent' ),
					'type'        => 'string',
					'context'     => array( 'view', 'edit' ),
				),
				'language'           => array(
					'description' => __( 'Cookie category language', 'webtoffee-cookie-consent' ),
					'type'        => 'string',
					'context'     => array( 'view', 'edit' ),
				),
				'description'        => array(
					'description' => __( 'Cookie category description.', 'webtoffee-cookie-consent' ),
					'type'        => 'object',
					'context'     => array( 'view', 'edit' ),
				),
				'prior_consent'      => array(
					'description' => __( 'Cookie type.', 'webtoffee-cookie-consent' ),
					'type'        => 'boolean',
					'context'     => array( 'view', 'edit' ),
				),
				'priority'           => array(
					'description' => __( 'Cookie type.', 'webtoffee-cookie-consent' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit' ),
				),
				'visibility'         => array(
					'description' => __( 'Show cookies on audit table or not', 'webtoffee-cookie-consent' ),
					'type'        => 'boolean',
					'context'     => array( 'view', 'edit' ),
				),
				'sell_personal_data' => array(
					'description' => __( 'Show cookies on audit table or not', 'webtoffee-cookie-consent' ),
					'type'        => 'boolean',
					'context'     => array( 'view', 'edit' ),
				),
				'cookies'            => array(
					'description' => __( 'Cookie category description.', 'webtoffee-cookie-consent' ),
					'type'        => 'object',
					'context'     => array( 'view' ),
				),

			),
		);
		return $this->add_additional_fields_schema( $schema );
	}
	/**
	 * Update no cookie script
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 * @return WP_Error|WP_REST_Response
	 */
	public function update_no_cookie_script( $request ) {
		$no_cookie_script = isset( $request['no_cookie_script'] ) ? $request['no_cookie_script'] : '';
		update_option( 'wcc_no_cookie_script', $no_cookie_script );
		return rest_ensure_response( array( 'no_cookie_script' => $no_cookie_script ) );
	}
} // End the class.
