<?php
/**
 * Class Api file.
 *
 * @package Gcm
 */

namespace WebToffee\CookieConsent\Lite\Admin\Modules\Gcm\Api;

use WP_REST_Server;
use WP_Error;
use WebToffee\CookieConsent\Lite\Includes\Rest_Controller;
use WebToffee\CookieConsent\Lite\Admin\Modules\Gcm\Includes\Controller;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

class Api extends Rest_Controller {

	/**
	 * Endpoint namespace.
	 *
	 * @var string
	 */
	protected $namespace = 'wcc/v1';
	/**
	 * Route base.
	 *
	 * @var string
	 */
	protected $rest_base = 'gcm';

	/**
	 * Constructor
	 */
	public function __construct() {
		add_action( 'rest_api_init', array( $this, 'register_routes' ), 10 );
	}
	/**
	 * Register the routes for gcm.
	 *
	 * @return void
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base,
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_items' ),
					'permission_callback' => array( $this, 'get_items_permissions_check' ),
					'args'                => $this->get_collection_params(),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/regions',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_regions' ),
					'permission_callback' => array( $this, 'get_items_permissions_check' ),
					'args'                => $this->get_endpoint_args_for_item_schema( WP_REST_Server::CREATABLE ),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/defaultconsents',
			array(
				array(
					'methods'             => 'POST',
					'callback'            => array( $this, 'default_consents' ),
					'permission_callback' => array( $this, 'create_item_permissions_check' ),
				),
			)
		);
	}

	/**
	 * Get a collection of items.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 * @return array
	 */
	public function get_items( $request ) {
		return get_option( 'wcc_gcm_default_consents', array() );
	}

	/**
	 * Get regions array
	 *
	 * @since 3.1.0
	 *
	 * @return WP_Error|WP_REST_Response
	 */
	public function get_regions() {
		return Controller::get_instance()->get_all_countries();
	}
	/**
	 * Update default consents
	 * @param WP_REST_Request $request Full details about the request.
	 * @since 3.1.0
	 * @return WP_Error|WP_REST_Response
	 */
	public function default_consents( $request ) {
		return Controller::get_instance()->update_default_consents( $request );
	}
	/**
	 * Get the Gcm's schema, conforming to JSON Schema.
	 *
	 * @return array
	 */
	public function get_item_schema() {
		$schema = array(
			'$schema'    => 'http://json-schema.org/draft-04/schema#',
			'title'      => 'gcm',
			'type'       => 'object',
			'properties' => array(
				'status'             => array(
					'description' => __( 'GCM status.', 'webtoffee-cookie-consent' ),
					'type'        => 'boolean',
					'context'     => array( 'view', 'edit' ),
				),
				'default_settings'   => array(
					'description' => __( 'Default settings.', 'webtoffee-cookie-consent' ),
					'type'        => 'array',
					'context'     => array( 'view', 'edit' ),
				),
				'wait_for_update'    => array(
					'description' => __( 'Wait for update.', 'webtoffee-cookie-consent' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit' ),
				),
				'url_passthrough'    => array(
					'description' => __( 'Pass ad click information through URLs.', 'webtoffee-cookie-consent' ),
					'type'        => 'boolean',
					'context'     => array( 'view', 'edit' ),
				),
				'ads_data_redaction' => array(
					'description' => __( 'Redact ads data.', 'webtoffee-cookie-consent' ),
					'type'        => 'boolean',
					'context'     => array( 'view', 'edit' ),
				),
			),
		);

		return $this->add_additional_fields_schema( $schema );
	}
}
