<?php
/**
 * Class Scanner file.
 *
 * @link       https://www.webtoffee.com/
 * @since      3.0.0
 * @package    WebToffee\CookieConsent\Lite\Admin\Modules\Scanner\Includes
 */

namespace WebToffee\CookieConsent\Lite\Admin\Modules\Scanner\Includes;

use WebToffee\CookieConsent\Lite\Includes\Store;
use WebToffee\CookieConsent\Lite\Admin\Modules\Scanner\Includes\Controller;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Handles Cookies Operation
 *
 * @class       Scanner
 * @version     3.0.0
 * @package     WebToffee
 */
class Scanner extends Store {

	/**
	 * Scanner controller class.
	 *
	 * @var object
	 */
	private $controller;

	/**
	 * Instance of the current class
	 *
	 * @var object
	 */
	private static $instance;

	/**
	 * Data array, with defaults.
	 *
	 * @var array
	 */
	protected $data = array(
		'scan_id'       => '',
		'scan_status'   => '',
		'scan_token'    => '',
		'date_created'  => '',
		'total_url'     => '',
		'total_cookies' => '',
		'category'      => '',
		'total_scripts' => '',
		'urls_scanned'  => '',
		'scan_history'  => '',
	);

	/**
	 * Constructor
	 *
	 * @param mixed $data ID or slug of the cookie.
	 */
	public function __construct( $data = '' ) {
		$this->controller = Controller::get_instance();
		parent::__construct( $data );
		if ( is_int( $data ) && 0 !== $data ) {
			$this->set_id( $data );
		}
		if ( isset( $data->id ) ) {
			$this->set_id( $data->id );
			$this->read_direct( $data );
		} else {
			$this->get_data_from_db();
		}
	}
	/**
	 * Return the current instance of the class
	 *
	 * @return object
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}
	/**
	 * Read directly from the data object given.
	 * Used for assigning data to object if it is alread fetched from API or DB.
	 *
	 * @param array|object $data Scanner data.
	 * @return void
	 */
	public function read_direct( $data ) {
		$this->set_data( $data );
	}

	/**
	 * Assign data to objects
	 *
	 * @param array|object $data Array of data.
	 * @return void
	 */
	public function set_data( $data ) {
		if ( isset( $data->id ) ) {
			$this->set_multi_item_data(
				array(
					'scan_id'       => $data->scan_id,
					'scan_status'   => $data->scan_status,
					'scan_token'    => $data->scan_token,
					'date_created'  => $data->date_created,
					'total_url'     => $data->total_url,
					'total_cookies' => $data->total_cookies,
					'total_scripts' => $data->total_scripts,
					'category'      => $data->category,
					'urls_scanned'  => $data->urls_scanned,
					'scan_history'  => $data->scan_history,
				)
			);
			$this->set_loaded( true );
		}
	}
	/**
	 * Read cookie data from database
	 *
	 * @param object $scanner Instance of Scanner.
	 * @return void
	 */
	public function read( $scanner ) {
		$scanner->set_defaults();
		$data = $this->controller->get_item( $scanner->get_id() );
		$this->set_data( $data );
	}

	/**
	 * Insert a new scanner on the database.
	 *
	 * @param object $scanner Consent scanner object.
	 * @return void
	 */
	public function create( $scanner ) {
		$this->controller->create_item( $scanner );
	}
	/**
	 * Update scanner data
	 *
	 * @param object $scanner Instance of Scanner.
	 * @return void
	 */
	public function update( $scanner ) {
		$this->controller->update_item( $scanner );
	}

	/**
	 * Check if an array is associative or indexed
	 *
	 * @param array $arr Input array.
	 * @return Boolean
	 */
	public static function array_has_key( $arr ) {
		if ( count( array_filter( array_keys( $arr ), 'is_string' ) ) === 0 ) {
			return false;
		}
		return true;
	}

	/**
	 * Sanitize the option values
	 *
	 * @param string $option The name of the option.
	 * @param string $value  The unsanitised value.
	 * @return string Sanitized value.
	 */
	public static function sanitize_option( $option, $value ) {
		switch ( $option ) {
			case 'status':
				$value = wcc_sanitize_bool( $value );
				break;
			case 'background-color':
				$value = wcc_sanitize_color( $value );
				break;
			default:
				$value = wcc_sanitize_text( $value );
				break;
		}
		return $value;
	}

	/**
	 * Sanitize the contents
	 *
	 * @param string $option The name of the option.
	 * @param string $value  The unsanitised value.
	 * @return string Sanitized value.
	 */
	public function sanitize_content( $option, $value ) {
		switch ( $option ) {
			case 'description':
				$value = wcc_sanitize_content( $value );
				break;
			default:
				$value = wcc_sanitize_text( $value );
				break;
		}
		return $value;
	}

	/**
	 * Get status of the item
	 *
	 * @since  3.0.0
	 * @return string
	 */
	public function get_scan_status() {
		return sanitize_text_field( $this->get_object_data( 'scan_status' ) );
	}

	/**
	 * Get token of the item
	 *
	 * @since  3.0.0
	 * @return string
	 */
	public function get_scan_token() {
		return $this->get_object_data( 'scan_token' );
	}

	/**
	 * Get id of the item
	 *
	 * @since  3.0.0
	 * @return scan_id
	 */
	public function get_scan_id() {
		return $this->get_object_data( 'scan_id' );
	}

	/**
	 * Get total url count of the item
	 *
	 * @since  3.0.0
	 * @return total_url
	 */
	public function get_total_url() {
		return $this->get_object_data( 'total_url' );
	}

	/**
	 * Get total cookies of the item
	 *
	 * @since  3.0.0
	 * @return total_cookies
	 */
	public function get_total_cookies() {
		return $this->get_object_data( 'total_cookies' );
	}

	/**
	 * Get total scripts of the item
	 *
	 * @since  3.0.0
	 * @return total_scripts
	 */
	public function get_total_scripts() {
		return $this->get_object_data( 'total_scripts' );
	}

	/**
	 * Get category of the item
	 *
	 * @since  3.0.0
	 * @return category
	 */
	public function get_category() {
		return $this->get_object_data( 'category' );
	}
	/**
	 * Get urls scanned
	 *
	 * @since  3.0.0
	 * @return urls_scanned
	 */
	public function get_urls_scanned() {
		return sanitize_text_field( $this->get_object_data( 'urls_scanned' ) );
	}
	/**
	 * Get scan_history of the item
	 *
	 * @since  3.0.0
	 * @return scan_history
	 */
	public function get_scan_history() {
		return sanitize_text_field( $this->get_object_data( 'scan_history' ) );
	}
	/**
	 * Set status of the item
	 *
	 * @since  3.0.0
	 * @return string
	 */
	public function set_scan_status( $scan_status ) {
		$this->set_object_data( 'scan_status', sanitize_text_field( $scan_status ) );
	}
	/**
	 * Set status of the item
	 *
	 * @since  3.0.0
	 * @return string
	 */
	public function set_scan_token( $scan_token ) {
		$this->set_object_data( 'scan_token', sanitize_text_field( $scan_token ) );
	}
	/**
	 * Set scan_id of the item
	 *
	 * @since  3.0.0
	 * @return scan_id
	 */
	public function set_scan_id( $scan_id ) {
		$this->set_object_data( 'scan_id', absint( $scan_id ) );
	}

	/**
	 *Set total url count of the item
	 *
	 * @since  3.0.0
	 * @return total_url
	 */
	public function set_total_url( $total_url ) {
		$this->set_object_data( 'total_url', absint( $total_url ) );
	}

	/**
	 * Set total cookies of the item
	 *
	 * @since  3.0.0
	 * @return total_cookies
	 */
	public function set_total_cookies( $total_cookies ) {
		$this->set_object_data( 'total_cookies', absint( $total_cookies ) );
	}

	/**
	 * Set total scripts of the item
	 *
	 * @since  3.0.0
	 * @return total_scripts
	 */
	public function set_total_scripts( $total_scripts ) {
		$this->set_object_data( 'total_scripts', absint( $total_scripts ) );
	}

	/**
	 * Set category of the item
	 *
	 * @since  3.0.0
	 * @return category
	 */
	public function set_category( $category ) {
		$this->set_object_data( 'category', absint( $category ) );
	}

	/**
	 * Set urls scanned
	 *
	 * @since  3.0.0
	 * @return total_scripts
	 */
	public function set_urls_scanned( $urls_scanned ) {
		$this->set_object_data( 'urls_scanned', $urls_scanned );
	}
	/**
	 * Set total scripts of the item
	 *
	 * @since  3.0.0
	 * @return total_scripts
	 */
	public function set_scan_history( $scan_history ) {
		$this->set_object_data( 'scan_history', sanitize_text_field( $scan_history ) );
	}
}
