<?php
/**
 * Class Banner file.
 *
 * @package WebToffee
 */

namespace WebToffee\CookieConsent\Lite\Admin\Modules\Settings\Includes;

use WebToffee\CookieConsent\Lite\Includes\Store;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Handles Cookies Operation
 *
 * @class       Settings
 * @version     3.0.0
 * @package     WebToffee
 */
class Settings extends Store {
	/**
	 * Data array, with defaults.
	 *
	 * @var array
	 */
	protected $data = array();

	/**
	 * Instance of the current class
	 *
	 * @var object
	 */
	private static $instance;

	/**
	 * Return the current instance of the class
	 *
	 * @return object
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}
	/**
	 * Constructor
	 */
	public function __construct() {
		$this->data = $this->get_defaults();
	}

	/**
	 * Get default plugin settings
	 *
	 * @return array
	 */
	public function get_defaults() {
		return array(
			'site'                 => array(
				'url'       => get_site_url(),
				'installed' => time(),
			),
			'api'                  => array(
				'token' => '',
			),
			'account'              => array(
				'email'       => '',
				'domain'      => '',
				'connected'   => false,
				'plan'        => 'free',
				'website_id'  => '',
				'website_key' => '',
			),
			'consent_logs'         => array(
				'status' => true,
			),
			'languages'            => array(
				'selected' => array( 'en' ),
				'default'  => 'en',
			),
			'banners'              => array(
				'selected' => array( 'GDPR' ),
			),
			'onboarding'           => array(
				'step' => 2,
			),
			'translation_modal'    => array(
				'show' => true,
			),
			'renew_date'           => array(
				'date' => '',
			),
			'google_consent_mode'  => array(
				'status' => true,
			),
			'url_passthrough'      => array(
				'status' => false,
			),
			'vendors'              => get_option( 'wcc_vendor_data', array() ),
			'gcm_debug_mode'       => array(
				'status' => false,
			),
			'gtag_behavior'        => array(
				'selected' => 'advanced',
			),
			'redact_ads_data'      => array(
				'status' => false,
			),
			'hide_banner_on_pages' => array(
				'selected' => array(),
			),
			'microsoft_clarity'    => array(
				'status' => false,
			),
			'microsoft_uet'        => array(
				'status' => false,
			),
			'gtm_template'         => array(
				'status' => false,
			),
		);
	}
	/**
	 * Get settings
	 *
	 * @param string $group Name of the group.
	 * @param string $key Name of the key.
	 * @return array
	 */
	public function get( $group = '', $key = '' ) {
		$settings = get_option( 'wcc_settings', $this->data );
		$settings = self::sanitize( $settings, $this->data );
		if ( empty( $key ) && empty( $group ) ) {
			return $settings;
		} elseif ( ! empty( $key ) && ! empty( $group ) ) {
			$settings = isset( $settings[ $group ] ) ? $settings[ $group ] : array();
			return isset( $settings[ $key ] ) ? $settings[ $key ] : array();
		} else {
			return isset( $settings[ $group ] ) ? $settings[ $group ] : array();
		}
	}

	/**
	 * Excludes a key from sanitizing multiple times.
	 *
	 * @return array
	 */
	public static function get_excludes() {
		return array(
			'selected',
		);
	}
	/**
	 * Update settings to database.
	 *
	 * @param array $data Array of settings data.
	 * @return void
	 */
	public function update( $data ) {
		$settings = self::sanitize( $data, $this->data );
		update_option( 'wcc_settings', $settings );
		do_action( 'wcc_after_update_settings', $settings );
	}

	/**
	 * Sanitize options
	 *
	 * @param array $settings Input settings array.
	 * @param array $defaults Default settings array.
	 * @return array
	 */
	public static function sanitize( $settings, $defaults ) {
		$result  = array();
		$exludes = self::get_excludes();
		foreach ( $defaults as $key => $data ) {
			$value = isset( $settings[ $key ] ) ? $settings[ $key ] : $data;
			if ( in_array( $key, $exludes, true ) ) {
				$result[ $key ] = self::sanitize_option( $key, $value );
				continue;
			}
			if ( is_array( $value ) ) {
				$result[ $key ] = self::sanitize( $value, $data );
			} else {
				if ( is_string( $key ) ) {
					$result[ $key ] = self::sanitize_option( $key, $value );
				}
			}
		}
		return $result;
	}

	/**
	 * Sanitize the option values
	 *
	 * @param string $option The name of the option.
	 * @param string $value  The unsanitised value.
	 * @return string Sanitized value.
	 */
	public static function sanitize_option( $option, $value ) {
		switch ( $option ) {
			case 'connected':
			case 'status':
			case 'connected':
				$value = wcc_sanitize_bool( $value );
				break;
			case 'installed':
			case 'step':
				$value = absint( $value );
				break;
			default:
				$value = wcc_sanitize_text( $value );
				break;
		}
		return $value;
	}

	// Getter Functions.

	/**
	 * Get account token for authentication.
	 *
	 * @return string
	 */
	public function get_token() {
		return $this->get( 'api', 'token' );
	}

	/**
	 * Get website ID
	 *
	 * @return string
	 */
	public function get_website_id() {
		return $this->get( 'account', 'website_id' );
	}
	/**
	 * Get website ID
	 *
	 * @return string
	 */
	public function get_plan() {
		return $this->get( 'account', 'plan' );
	}
	/**
	 * Get the website key
	 *
	 * @return string
	 */
	public function get_website_key() {
		return $this->get( 'account', 'website_key' );
	}
	/**
	 * Get current site URL.
	 *
	 * @return string
	 */
	public function get_url() {
		return $this->get( 'site', 'url' );
	}
	/**
	 * Get show translation modal status
	 *
	 * @return boolean
	 */
	public function get_translation_modal_status() {
		return (bool) $this->get( 'translation_modal', 'show' );
	}
	/**
	 * Get consent log status
	 *
	 * @return boolean
	 */
	public function get_consent_log_status() {
		return (bool) $this->get( 'consent_logs', 'status' );
	}

	/**
	 * Returns the default language code
	 *
	 * @return string
	 */
	public function get_default_language() {
		return sanitize_text_field( $this->get( 'languages', 'default' ) );
	}

	/**
	 * Returns the selected languages.
	 *
	 * @return array
	 */
	public function get_selected_languages() {
		return wcc_sanitize_text( $this->get( 'languages', 'selected' ) );
	}

	/**
	 * First installed date of the plugin.
	 *
	 * @return string
	 */
	public function get_installed_date() {
		return $this->get( 'site', 'installed' );
	}

	/**
	 * Returns the selected laws.
	 *
	 * @return array
	 */
	public function get_selected_laws() {
		return wcc_sanitize_text( $this->get( 'banners', 'selected' ) );
	}

	/**
	 * Returns the site language code
	 *
	 * @return string
	 */
	public function get_site_language() {
		$language_code = 'en';
		$site_language = null;
		$language      = get_locale();
		$parts         = explode( '_', $language );
		if ( count( $parts ) > 0 ) {
			$language_code = $parts[0];
		}
		$site_language = $language_code;
		return apply_filters( 'wcc_site_language', $site_language );
	}

	/**
	 * Checks whether site is multilingual or
	 *
	 * @return string
	 */
	public function is_site_multilingual() {
		return apply_filters( 'wcc_is_mutlilingual_site', wcc_i18n_is_multilingual() );
	}
	/**
	 * Returns the policy page id
	 *
	 * @return string
	 */
	public function get_privacy_policy() {
		return sanitize_text_field( get_option( 'wcc_policy_id', '' ) );
	}
	/**
	 * Returns the consent renewed date
	 *
	 * @return string
	 */
	public function get_renew_date() {
		return sanitize_text_field( $this->get( 'renew_date', 'date' ) );
	}
	/**
	 * Get show Migration notice status
	 *
	 * @return boolean
	 */
	public function get_migration_notice_status() {
		return (bool) get_option( 'wt_cli_cookie_db_version', false ) && ! get_option( 'wcc_migration_notice_dismissed', false );
	}
	/**
	 * Get show renew consent notice status
	 *
	 * @return boolean
	 */
	public function get_renew_consent_notice_status() {
		return (bool) get_option( 'wcc_show_renew_consent_notice', false );
	}
	/**
	 * Get no cookie scripts to block
	 * @since 3.2.0
	 * @return array
	 */
	public function get_no_cookie_scripts_to_block() {
		return wcc_sanitize_text( get_option( 'wcc_no_cookie_script', array() ) );
	}
	/**
	 * Get review reminder data
	 * @since 3.2.0
	 * @return array
	 */
	public function get_review_reminder_data() {
		return wcc_sanitize_text( get_option( 'wcc_review_reminder', array() ) );
	}
}
