import wccRest from "@/helpers/api";


const banners = {
    namespaced: true,
    state: {
        items: false,
        current: false,
        preview: false,
        presets: [],
        configs: [],
        template: [],
        errors: {},
    },

    getters: {
        /**
         * Get the all settings object.
         *
         * @param {object} state Current state.
         *
         * @return {*}
         */
        getBanners: (state) => {
            return state.items
        },
        getCurrentBanner: (state) => {
            return state.current
        },
        getPreview: (state) => {
            return state.preview
        },
        getPresets: (state) => {
            return state.presets
        },
        getConfigs: (state) => {
            return state.configs
        },
        getTemplate: (state) => {
            return state.template
        },
        getErrors: (state) => {
            return state.errors
        },
    },

    mutations: {
        /**
         * Update a single value in store.
         *
         * Updates the value only on store.
         * Call updateSettings mutation to store the values in db.
         *
         * @param {object} state Current state.
         * @param {object} data Data to update.
         */
        setBanners: (state, data) => {
            state.items = data
        },
        setCurrentBanner: (state, data) => {
            state.current = data
        },
        setPreview: (state, data) => {
            state.preview = data
        },
        setPresets: (state, data) => {
            state.presets = data
        },
        setConfigs: (state, data) => {
            state.configs = data
        },
        setTemplate: (state, data) => {
            state.template = data
        },

        /**
         * Update the whole settings data in state.
         *
         * This will replace the current state with the
         * data provided.
         *
         * @param {object} state Current state.
         * @param {object} data Custom params.
         */
        updateBanners: (state, data) => {
            state.items = data
        },
        setErrors: (state, data) => {
            state.errors = data
        },
    },

    actions: {
        /**
         * Re-initialize the settings store forcefully.
         *
         * We need to call the API and get the settings,
         * then update the store state with the new values.
         *
         * @param commit Commit
         * @param {object} data Custom params.
         *
         * @return {Promise<void>}
         */
        reInit: async ({ commit }) => {
            try {
                const response = await wccRest.get({
                    path: "banners",
                })
                if (response) {
                    const banners = [];
                    response.forEach(function (item) {
                        let banner = {};
                        banner.id = item.id || '';
                        banner.name = item.name || '';
                        banner.slug = item.slug || '';
                        banner.default = item.default || false
                        banner.status = item.status || false
                        banner.properties = item.properties || {};
                        banner.contents = item.contents || {};
                        banners.push(banner);
                    });
                    commit('updateBanners', banners);
                }
            } catch (error) {
                console.log(error);
            }

        },

        /**
         * Set a single option value after validation.
         *
         * Make sure all required items are provided, then
         * call the mutation.
         *
         * @param commit Commit.
         * @param {object} data Custom params.
         */
        setBanners: async ({ commit }, data = {}) => {
            commit('setBanners', data)
        },
        setCurrentBanner: async ({ commit }, data = {}) => {
            commit('setCurrentBanner', data)
        },
        setPreview: async ({ commit }, data = {}) => {
            commit('setPreview', data)
        },
        setPresets: async ({ commit }, data = {}) => {
            commit('setPresets', data)
        },
        setConfigs: async ({ commit }, data = {}) => {
            commit('setConfigs', data)
        },
        setTemplate: async ({ commit }, data = {}) => {
            commit('setTemplate', data)
        },
        /**
         * Update the settings values in db.
         *
         * Use the API and update the whole values.
         * Only do this when required.
         *
         * @param {object} Commit and State.
         * @param {object} data Custom params.
         *
         * @return {boolean}
         */
        saveBanner: async ({ dispatch }, data = {}) => {
            let success = false;
            if (!Object.prototype.hasOwnProperty.call(data, 'banner')) return;
            const banner = data.banner;
            const query = data.params && data.params || {};
            try {
                const response = await wccRest.put({
                    path: `/banners/${banner.id}`,
                    data: banner,
                    params: query,
                });
                if (response.id) {
                    success = true;
                    dispatch('reInit');
                }
            } catch (error) {
                console.log(error)
            }

            return success
        },
        setErrors: async ({ commit }, data = {}) => {
            commit('setErrors', data)
        },
    },
}
export default banners