<?php
/**
 * Module: DynamicContentOptionPostDate class.
 *
 * @package Divi
 * @since ??
 */

namespace ET\Builder\Packages\Module\Layout\Components\DynamicContent;

if ( ! defined( 'ABSPATH' ) ) {
	die( 'Direct access forbidden.' );
}

use ET\Builder\Packages\Module\Layout\Components\DynamicContent\DynamicContentUtils;
use ET\Builder\Packages\Module\Layout\Components\DynamicContent\DynamicContentElements;

/**
 * Module: DynamicContentOptionPostDate class.
 *
 * @since ??
 */
class DynamicContentOptionPostDate extends DynamicContentOptionBase implements DynamicContentOptionInterface {

	/**
	 * Get the name of the post date option.
	 *
	 * @since ??
	 *
	 * @return string The name of the post date option.
	 */
	public function get_name(): string {
		return 'post_date';
	}

	/**
	 * Get the label for the post date option.
	 *
	 * This function retrieves the localized label for the post date option,
	 * which is used to describe the post date in user interfaces.
	 *
	 * @since ??
	 *
	 * @return string The label for the post date option.
	 */
	public function get_label(): string {
		// Translators: %1$s: Post type name.
		return __( '%1$s Publish Date', 'et_builder_5' );
	}

	/**
	 * Callback for registering post date option .
	 *
	 * This function is a callback for the `divi_module_dynamic_content_options` filter .
	 * This function is used to register options for post date by adding them to the options array passed to the function .
	 * It checks if the current module's name exists as a key in the options array.
	 * If not, it adds the module's name as a key and the specific options for that module as the value.
	 *
	 * @since ??
	 *
	 * @param array  $options The options array to be registered.
	 * @param int    $post_id The post ID.
	 * @param string $context The context in which the options are retrieved e.g `edit`, `display`.
	 *
	 * @return array The options array.
	 */
	public function register_option_callback( array $options, int $post_id, string $context ): array {
		$custom_fields = DynamicContentUtils::get_date_format_fields();

		if ( ! isset( $options[ $this->get_name() ] ) ) {
			$before_after_fields = array_merge(
				array_slice( DynamicContentUtils::get_common_loop_fields(), 0, 2, true ),
				$custom_fields
			);

			$options[ $this->get_name() ] = [
				'id'     => $this->get_name(),
				'label'  => esc_html( sprintf( $this->get_label(), DynamicContentUtils::get_post_type_label( $post_id ) ) ),
				'type'   => 'text',
				'custom' => false,
				'group'  => 'Default',
				'fields' => $before_after_fields,
			];

			$loop_fields = array_merge( DynamicContentUtils::get_common_loop_fields(), $custom_fields );

			$options[ 'loop_' . $this->get_name() ] = [
				'id'     => 'loop_' . $this->get_name(),
				'label'  => esc_html( sprintf( $this->get_label(), 'Loop' ) ),
				'type'   => 'text',
				'custom' => false,
				'group'  => 'Loop',
				'fields' => $loop_fields,
			];
		}

		if ( ! isset( $options[ 'loop_product_' . $this->get_name() ] ) && et_is_woocommerce_plugin_active() ) {
			$options[ 'loop_product_' . $this->get_name() ] = [
				'id'     => 'loop_product_' . $this->get_name(),
				'label'  => esc_html( sprintf( $this->get_label(), __( 'Loop Product', 'et_builder_5' ) ) ),
				'type'   => 'text',
				'custom' => false,
				'group'  => 'Loop Product',
				'fields' => $loop_fields,
			];
		}

		return $options;
	}

	/**
	 * Render callback for post date option.
	 *
	 * Retrieves the value of post date option based on the provided arguments and settings.
	 * This is a callback for `divi_module_dynamic_content_resolved_value` filter.
	 *
	 * @since ??
	 *
	 * @param mixed $value     The current value of the post date option.
	 * @param array $data_args {
	 *     Optional. An array of arguments for retrieving the post date.
	 *     Default `[]`.
	 *
	 *     @type string  $name       Optional. Option name. Default empty string.
	 *     @type array   $settings   Optional. Option settings. Default `[]`.
	 *     @type integer $post_id    Optional. Post Id. Default `null`.
	 * }
	 *
	 * @return string The formatted value of the post date option.
	 *
	 * @example:
	 * ```php
	 *  $element = new MyDynamicContentElement();
	 *
	 *  // Render the element with a specific value and data arguments.
	 *  $html = $element->render_callback( $value, [
	 *      'name'     => 'my_element',
	 *      'settings' => [
	 *          'post_id' => 123,
	 *          'foo'     => 'bar',
	 *      ],
	 *      'post_id'  => 456,
	 *  ] );
	 * ```
	 */
	public function render_callback( $value, array $data_args = [] ): string {
		$name     = $data_args['name'] ?? '';
		$settings = $data_args['settings'] ?? [];
		$post_id  = $data_args['post_id'] ?? null;

		if ( $name !== $this->get_name() ) {
			return $value;
		}

		$post = is_int( $post_id ) && 0 !== $post_id ? get_post( $post_id ) : false;

		if ( $post ) {
			$format        = $settings['date_format'] ?? 'default';
			$custom_format = $settings['custom_date_format'] ?? '';

			if ( 'custom' === $format ) {
				// Convert double backslashes to single for PHP date functions.
				$format = str_replace( '\\\\', '\\', $custom_format );
			} elseif ( 'default' === $format ) {
				$format = '';
			}

			$value = esc_html( get_the_date( $format, $post_id ) );
		}

		return DynamicContentElements::get_wrapper_element(
			[
				'post_id'  => $post_id,
				'name'     => $name,
				'value'    => $value,
				'settings' => $settings,
			]
		);
	}
}
